<?php
// user_customers.php
require_once 'config.php';
redirectIfNotLoggedIn();
redirectIfNotUser();

$message = '';
$error = '';
$customers = [];

try {
    $db = getDB();
    
    // Get customers for this store
    $stmt = $db->prepare("SELECT * FROM customers WHERE store_user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$_SESSION['user_id']]);
    $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Handle customer actions
    if (isset($_GET['action']) && isset($_GET['customer_id'])) {
        $action = $_GET['action'];
        $customerId = $_GET['customer_id'];
        
        switch($action) {
            case 'activate':
                $updateStmt = $db->prepare("UPDATE customers SET status = 'active' WHERE id = ? AND store_user_id = ?");
                $updateStmt->execute([$customerId, $_SESSION['user_id']]);
                $message = "Customer activated successfully!";
                break;
            case 'suspend':
                $updateStmt = $db->prepare("UPDATE customers SET status = 'suspended' WHERE id = ? AND store_user_id = ?");
                $updateStmt->execute([$customerId, $_SESSION['user_id']]);
                $message = "Customer suspended successfully!";
                break;
            case 'delete':
                $deleteStmt = $db->prepare("DELETE FROM customers WHERE id = ? AND store_user_id = ?");
                $deleteStmt->execute([$customerId, $_SESSION['user_id']]);
                $message = "Customer deleted successfully!";
                break;
        }
        
        // Refresh customers list
        $stmt->execute([$_SESSION['user_id']]);
        $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
} catch(Exception $e) {
    $error = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Customers - VPS Manager</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .message { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background-color: #f8f9fa; font-weight: bold; }
        .btn { display: inline-block; padding: 6px 12px; margin: 2px; border: none; border-radius: 4px; cursor: pointer; text-decoration: none; font-size: 12px; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: black; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-info { background: #17a2b8; color: white; }
        .status-active { color: #28a745; font-weight: bold; }
        .status-suspended { color: #dc3545; font-weight: bold; }
        .status-inactive { color: #6c757d; }
        .nav-links { margin-bottom: 20px; }
        .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
    </style>
</head>
<body>
    <div class="nav-links">
        <a href="user_vps.php">My VPS</a> | 
        <a href="user_customers.php">My Customers</a> | 
        <a href="user_store_settings.php">Store Settings</a> | 
        <a href="logout.php">Logout</a>
    </div>

    <div class="header">
        <h1>My Customers</h1>
        <div>
            <span>Total Customers: <?php echo count($customers); ?></span>
        </div>
    </div>
    
    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if (empty($customers)): ?>
        <p>No customers registered in your store yet.</p>
        <p>Customers can register at: http://your-store-domain.com/store_dashboard.php</p>
    <?php else: ?>
        <table>
            <thead>
                <tr>
                    <th>Username</th>
                    <th>Email</th>
                    <th>Name</th>
                    <th>Status</th>
                    <th>Registered</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($customers as $customer): ?>
                <tr>
                    <td><?php echo htmlspecialchars($customer['username']); ?></td>
                    <td><?php echo htmlspecialchars($customer['email']); ?></td>
                    <td>
                        <?php 
                        $name = trim($customer['first_name'] . ' ' . $customer['last_name']);
                        echo $name ? htmlspecialchars($name) : 'N/A';
                        ?>
                    </td>
                    <td class="status-<?php echo strtolower($customer['status']); ?>">
                        <?php echo htmlspecialchars($customer['status']); ?>
                    </td>
                    <td><?php echo date('M j, Y H:i', strtotime($customer['created_at'])); ?></td>
                    <td>
                        <?php if ($customer['status'] === 'active'): ?>
                            <a href="user_customers.php?action=suspend&customer_id=<?php echo $customer['id']; ?>" 
                               class="btn btn-warning" 
                               onclick="return confirm('Suspend this customer?')">Suspend</a>
                        <?php else: ?>
                            <a href="user_customers.php?action=activate&customer_id=<?php echo $customer['id']; ?>" 
                               class="btn btn-success" 
                               onclick="return confirm('Activate this customer?')">Activate</a>
                        <?php endif; ?>
                        <a href="user_customers.php?action=delete&customer_id=<?php echo $customer['id']; ?>" 
                           class="btn btn-danger" 
                           onclick="return confirm('Delete this customer? This cannot be undone.')">Delete</a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</body>
</html>