<?php
// store_dashboard.php
require_once 'config.php';

// Detect domain and get store user
$current_domain = detectStoreDomain();
$store_user = getStoreUserByDomain($current_domain);

if (!$store_user) {
    // If no store found for this domain, redirect to main site
    header("Location: " . Config::BASE_URL . "/index.php");
    exit;
}

// Store-specific login/register logic
$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'login') {
        $username = trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        
        if (empty($username) || empty($password)) {
            $error = "Please enter both username and password!";
        } else {
            try {
                $db = getDB();
                $stmt = $db->prepare("
                    SELECT c.* 
                    FROM customers c 
                    WHERE c.username = ? AND c.store_user_id = ? AND c.status = 'active'
                ");
                $stmt->execute([$username, $store_user['id']]);
                $customer = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($customer && password_verify($password, $customer['password_hash'])) {
                    // Login successful
                    $_SESSION['user_id'] = $customer['id'];
                    $_SESSION['username'] = $customer['username'];
                    $_SESSION['user_role'] = 'customer';
                    $_SESSION['store_user_id'] = $store_user['id'];
                    $_SESSION['store_domain'] = $current_domain;
                    
                    header("Location: customer_vps.php");
                    exit;
                } else {
                    $error = "Invalid username or password!";
                }
                
            } catch(PDOException $e) {
                $error = "Database error: " . $e->getMessage();
            }
        }
    } elseif ($action === 'register') {
        $username = trim($_POST['username'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        $first_name = trim($_POST['first_name'] ?? '');
        $last_name = trim($_POST['last_name'] ?? '');
        
        if (empty($username) || empty($email) || empty($password)) {
            $error = "Username, email and password are required!";
        } elseif ($password !== $confirm_password) {
            $error = "Passwords do not match!";
        } elseif (strlen($password) < 6) {
            $error = "Password must be at least 6 characters long!";
        } else {
            try {
                $db = getDB();
                
                // Check if username or email already exists in this store
                $checkStmt = $db->prepare("
                    SELECT id FROM customers 
                    WHERE (username = ? OR email = ?) AND store_user_id = ?
                ");
                $checkStmt->execute([$username, $email, $store_user['id']]);
                
                if ($checkStmt->fetch()) {
                    $error = "Username or email already exists in this store!";
                } else {
                    // Create new customer
                    $password_hash = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $db->prepare("
                        INSERT INTO customers (store_user_id, username, email, password_hash, first_name, last_name) 
                        VALUES (?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$store_user['id'], $username, $email, $password_hash, $first_name, $last_name]);
                    
                    $message = "Registration successful! You can now login.";
                }
                
            } catch(PDOException $e) {
                $error = "Database error: " . $e->getMessage();
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($store_user['store_name']); ?> - VPS Store</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 0; padding: 0; background: #f8f9fa; }
        .header { background: #343a40; color: white; padding: 20px; text-align: center; }
        .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        .store-info { text-align: center; margin-bottom: 30px; }
        .auth-forms { display: grid; grid-template-columns: 1fr 1fr; gap: 30px; }
        .form-container { background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; }
        input[type="text"], input[type="email"], input[type="password"] { 
            width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; 
        }
        button { background: #007bff; color: white; padding: 12px 20px; border: none; border-radius: 4px; cursor: pointer; width: 100%; }
        .message { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .form-toggle { text-align: center; margin-top: 20px; }
        .hidden { display: none; }
    </style>
</head>
<body>
    <div class="header">
        <h1><?php echo htmlspecialchars($store_user['store_name']); ?></h1>
        <p>VPS Hosting Services</p>
    </div>
    
    <div class="container">
        <div class="store-info">
            <h2>Welcome to <?php echo htmlspecialchars($store_user['store_name']); ?></h2>
            <p>Manage your VPS instances with ease</p>
        </div>
        
        <?php if ($message): ?>
            <div class="message success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="message error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="auth-forms">
            <!-- Login Form -->
            <div class="form-container">
                <h3>Customer Login</h3>
                <form method="POST">
                    <input type="hidden" name="action" value="login">
                    <div class="form-group">
                        <label for="login_username">Username:</label>
                        <input type="text" id="login_username" name="username" required>
                    </div>
                    <div class="form-group">
                        <label for="login_password">Password:</label>
                        <input type="password" id="login_password" name="password" required>
                    </div>
                    <button type="submit">Login</button>
                </form>
            </div>
            
            <!-- Registration Form -->
            <div class="form-container">
                <h3>Customer Registration</h3>
                <form method="POST">
                    <input type="hidden" name="action" value="register">
                    <div class="form-group">
                        <label for="reg_username">Username:</label>
                        <input type="text" id="reg_username" name="username" required>
                    </div>
                    <div class="form-group">
                        <label for="reg_email">Email:</label>
                        <input type="email" id="reg_email" name="email" required>
                    </div>
                    <div class="form-group">
                        <label for="reg_first_name">First Name:</label>
                        <input type="text" id="reg_first_name" name="first_name">
                    </div>
                    <div class="form-group">
                        <label for="reg_last_name">Last Name:</label>
                        <input type="text" id="reg_last_name" name="last_name">
                    </div>
                    <div class="form-group">
                        <label for="reg_password">Password:</label>
                        <input type="password" id="reg_password" name="password" required>
                    </div>
                    <div class="form-group">
                        <label for="reg_confirm_password">Confirm Password:</label>
                        <input type="password" id="reg_confirm_password" name="confirm_password" required>
                    </div>
                    <button type="submit">Register</button>
                </form>
            </div>
        </div>
        
        <div style="text-align: center; margin-top: 30px; color: #6c757d;">
            <p>Powered by VPS Manager</p>
        </div>
    </div>
</body>
</html>