<?php
// register_store_user.php
require_once 'config.php';

// Only admins can register store users
redirectIfNotLoggedIn();
redirectIfNotAdmin();

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $store_name = trim($_POST['store_name'] ?? '');
    $store_domain = trim($_POST['store_domain'] ?? '');
    
    // Validation
    if (empty($username) || empty($email) || empty($password) || empty($store_name) || empty($store_domain)) {
        $error = "All fields are required!";
    } elseif ($password !== $confirm_password) {
        $error = "Passwords do not match!";
    } elseif (strlen($password) < 6) {
        $error = "Password must be at least 6 characters long!";
    } else {
        try {
            $db = getDB();
            
            // Check if username, email or domain already exists
            $checkStmt = $db->prepare("
                SELECT u.id FROM users u 
                LEFT JOIN user_store_settings uss ON u.id = uss.user_id 
                WHERE u.username = ? OR u.email = ? OR uss.store_domain = ?
            ");
            $checkStmt->execute([$username, $email, $store_domain]);
            
            if ($checkStmt->fetch()) {
                $error = "Username, email or domain already exists!";
            } else {
                // Create new store user
                $password_hash = password_hash($password, PASSWORD_DEFAULT);
                $db->beginTransaction();
                
                // Insert user
                $userStmt = $db->prepare("INSERT INTO users (username, email, password_hash, role) VALUES (?, ?, ?, 'user')");
                $userStmt->execute([$username, $email, $password_hash]);
                $user_id = $db->lastInsertId();
                
                // Insert store settings
                $storeStmt = $db->prepare("INSERT INTO user_store_settings (user_id, store_name, store_domain) VALUES (?, ?, ?)");
                $storeStmt->execute([$user_id, $store_name, $store_domain]);
                
                $db->commit();
                $message = "Store user created successfully! Store domain: " . htmlspecialchars($store_domain);
            }
            
        } catch(PDOException $e) {
            $db->rollBack();
            $error = "Database error: " . $e->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register Store User - VPS Manager</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 500px; margin: 50px auto; padding: 20px; }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; }
        input[type="text"], input[type="email"], input[type="password"] { 
            width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; 
        }
        button { background: #007bff; color: white; padding: 10px 20px; border: none; border-radius: 4px; cursor: pointer; width: 100%; }
        .message { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .nav-links { text-align: center; margin-top: 20px; }
    </style>
</head>
<body>
    <div class="nav-links">
        <a href="view_vps.php">VPS List</a> | 
        <a href="user_management.php">User Management</a> | 
        <a href="logout.php">Logout</a>
    </div>

    <h1>Register Store User</h1>
    
    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <form method="POST">
        <div class="form-group">
            <label for="username">Username:</label>
            <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($username ?? ''); ?>" required>
        </div>
        
        <div class="form-group">
            <label for="email">Email:</label>
            <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($email ?? ''); ?>" required>
        </div>
        
        <div class="form-group">
            <label for="store_name">Store Name:</label>
            <input type="text" id="store_name" name="store_name" value="<?php echo htmlspecialchars($store_name ?? ''); ?>" required>
        </div>
        
        <div class="form-group">
            <label for="store_domain">Store Domain:</label>
            <input type="text" id="store_domain" name="store_domain" value="<?php echo htmlspecialchars($store_domain ?? ''); ?>" required>
            <small>e.g., mystore.example.com</small>
        </div>
        
        <div class="form-group">
            <label for="password">Password:</label>
            <input type="password" id="password" name="password" required>
        </div>
        
        <div class="form-group">
            <label for="confirm_password">Confirm Password:</label>
            <input type="password" id="confirm_password" name="confirm_password" required>
        </div>
        
        <button type="submit">Create Store User</button>
    </form>
</body>
</html>