<?php
// ContaboAPI.php
require_once 'config.php';

class ContaboAPI {
    private $access_token;
    private $settings;
    
    public function __construct() {
        $this->loadSettings();
    }
    
    private function loadSettings() {
        $db = getDB();
        $stmt = $db->query("SELECT * FROM contabo_settings LIMIT 1");
        $this->settings = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$this->settings) {
            throw new Exception("Contabo settings not found. Please configure them first.");
        }
    }
    
    // Generate a valid UUID v4
    private function generateUuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
    
    public function authenticate() {
        $auth_data = [
            'client_id' => $this->settings['client_id'],
            'client_secret' => $this->settings['client_secret'],
            'username' => $this->settings['api_user'],
            'password' => $this->settings['api_password'],
            'grant_type' => 'password'
        ];
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_AUTH_URL,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($auth_data),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded'
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200) {
            throw new Exception("Authentication failed: HTTP $http_code - " . $response);
        }
        
        $data = json_decode($response, true);
        $this->access_token = $data['access_token'];
        
        return $this->access_token;
    }
    
    // Add cancel instance method
    public function cancelInstance($instanceId, $cancelDate = null) {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        // Default to 30 days from now if no date provided
        if (!$cancelDate) {
            $cancelDate = date('Y-m-d', strtotime('+30 days'));
        }
        
        $data = [
            'cancelDate' => $cancelDate
        ];
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/actions/cancel',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 201) {
            throw new Exception("Failed to cancel instance: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    // Add remove cancellation method
    public function removeCancellation($instanceId) {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        // Note: Contabo API might not have a direct "remove cancellation" endpoint
        // This would typically be done by updating the instance to remove cancellation
        // For now, we'll throw an informative error
        throw new Exception("Remove cancellation functionality is not directly available in the Contabo API. Please contact Contabo support to remove cancellation.");
    }
    
    // Add rollback snapshot method
    public function rollbackSnapshot($instanceId, $snapshotId) {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $data = new stdClass(); // Empty object as required by API
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/snapshots/' . $snapshotId . '/actions/rollback',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200) {
            throw new Exception("Failed to rollback snapshot: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    public function getInstances() {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200) {
            throw new Exception("Failed to fetch instances: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    public function getInstanceDetails($instanceId) {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200) {
            throw new Exception("Failed to fetch instance details: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    public function getAvailableImages() {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/images',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200) {
            throw new Exception("Failed to fetch images: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    // Get popular OS images with better filtering
    public function getPopularOSImages() {
        $images_data = $this->getAvailableImages();
        $popular_images = [];
        
        if (isset($images_data['data']) && is_array($images_data['data'])) {
            foreach ($images_data['data'] as $image) {
                $name = strtolower($image['name'] ?? '');
                $osType = strtolower($image['osType'] ?? '');
                
                // Filter for popular distributions
                if (strpos($name, 'ubuntu') !== false || 
                    strpos($name, 'debian') !== false ||
                    strpos($name, 'centos') !== false ||
                    strpos($name, 'alma') !== false || // AlmaLinux
                    strpos($name, 'rocky') !== false ||
                    strpos($name, 'fedora') !== false ||
                    strpos($name, 'windows') !== false ||
                    strpos($name, 'freebsd') !== false) {
                    $popular_images[] = $image;
                }
            }
        }
        
        return $popular_images;
    }
    
    public function reinstallInstance($instanceId, $imageId, $sshKeys = [], $rootPassword = null, $userData = '', $defaultUser = 'admin') {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $data = [
            'imageId' => $imageId,
            'defaultUser' => $defaultUser
        ];
        
        if (!empty($sshKeys)) {
            $data['sshKeys'] = $sshKeys;
        }
        
        if ($rootPassword) {
            $data['rootPassword'] = $rootPassword;
        }
        
        if (!empty($userData)) {
            $data['userData'] = $userData;
        }
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/actions/reinstall',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200) {
            throw new Exception("Failed to reinstall instance: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    public function createSnapshot($instanceId, $name, $description = '') {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $data = [
            'name' => $name,
            'description' => $description
        ];
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/snapshots',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 201) {
            throw new Exception("Failed to create snapshot: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    public function getSnapshots($instanceId) {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/snapshots',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200) {
            throw new Exception("Failed to fetch snapshots: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    public function deleteSnapshot($instanceId, $snapshotId) {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/snapshots/' . $snapshotId,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'DELETE',
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 204 && $http_code !== 200) {
            throw new Exception("Failed to delete snapshot: HTTP $http_code - " . $response);
        }
        
        return true;
    }
    
    public function resetPassword($instanceId, $sshKeys = [], $rootPassword = null, $userData = '') {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $data = [];
        
        if (!empty($sshKeys)) {
            $data['sshKeys'] = $sshKeys;
        }
        
        if ($rootPassword) {
            $data['rootPassword'] = $rootPassword;
        }
        
        if (!empty($userData)) {
            $data['userData'] = $userData;
        }
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/actions/resetPassword',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 201) {
            throw new Exception("Failed to reset password: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    public function rescueInstance($instanceId, $rootPassword = null, $sshKeys = [], $userData = '') {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $data = [];
        
        if ($rootPassword) {
            $data['rootPassword'] = $rootPassword;
        }
        
        if (!empty($sshKeys)) {
            $data['sshKeys'] = $sshKeys;
        }
        
        if (!empty($userData)) {
            $data['userData'] = $userData;
        }
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/actions/rescue',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 201) {
            throw new Exception("Failed to boot in rescue mode: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    public function startInstance($instanceId) {
        return $this->performInstanceAction($instanceId, 'start');
    }
    
    public function stopInstance($instanceId) {
        return $this->performInstanceAction($instanceId, 'stop');
    }
    
    public function restartInstance($instanceId) {
        return $this->performInstanceAction($instanceId, 'restart');
    }
    
    public function shutdownInstance($instanceId) {
        return $this->performInstanceAction($instanceId, 'shutdown');
    }
    
    private function performInstanceAction($instanceId, $action) {
        if (!$this->access_token) {
            $this->authenticate();
        }
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/actions/' . $action,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->access_token,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 201) {
            throw new Exception("Failed to $action instance: HTTP $http_code - " . $response);
        }
        
        return json_decode($response, true);
    }
    
    public function syncInstancesToDB() {
        try {
            $instances_data = $this->getInstances();
            $db = getDB();
            
            // Clear existing instances
            $db->query("TRUNCATE TABLE vps_instances");
            
            $stmt = $db->prepare("
                INSERT INTO vps_instances 
                (instance_id, name, status, ip_address, created_date, region, image_id, product_type) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            foreach ($instances_data['data'] as $instance) {
                $ip_address = '';
                if (isset($instance['ipConfig']['v4']['ip']) && !empty($instance['ipConfig']['v4']['ip'])) {
                    $ip_address = $instance['ipConfig']['v4']['ip'][0];
                }
                
                $stmt->execute([
                    $instance['instanceId'],
                    $instance['name'],
                    $instance['status'],
                    $ip_address,
                    date('Y-m-d H:i:s', strtotime($instance['createdDate'])),
                    $instance['region'],
                    $instance['imageId'],
                    $instance['productType']
                ]);
            }
            
            return count($instances_data['data']);
            
        } catch(Exception $e) {
            throw new Exception("Sync failed: " . $e->getMessage());
        }
    }
    
    // Log actions to database
    public function logAction($instanceId, $actionType, $actionData = '') {
        try {
            $db = getDB();
            $stmt = $db->prepare("
                INSERT INTO instance_actions_log 
                (instance_id, action_type, action_data, performed_by) 
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([
                $instanceId,
                $actionType,
                $actionData,
                $_SESSION['user_id']
            ]);
            return true;
        } catch(Exception $e) {
            // Don't throw error if logging fails
            error_log("Failed to log action: " . $e->getMessage());
            return false;
        }
    }
    
    // Helper function to fix IP address display
    public function formatIPAddresses($ipConfig) {
        $formatted = [
            'ipv4' => [],
            'ipv6' => []
        ];
        
        if (isset($ipConfig['v4']['ip'])) {
            if (is_array($ipConfig['v4']['ip'])) {
                $formatted['ipv4'] = $ipConfig['v4']['ip'];
            } elseif (is_string($ipConfig['v4']['ip'])) {
                $formatted['ipv4'] = [$ipConfig['v4']['ip']];
            }
        }
        
        if (isset($ipConfig['v6']['ip'])) {
            if (is_array($ipConfig['v6']['ip'])) {
                $formatted['ipv6'] = $ipConfig['v6']['ip'];
            } elseif (is_string($ipConfig['v6']['ip'])) {
                $formatted['ipv6'] = [$ipConfig['v6']['ip']];
            }
        }
        
        return $formatted;
    }
}
?>